
#define STB_IMAGE_IMPLEMENTATION
#include "stb-master/stb_image.h"

// Types.

	struct Pixel {
		u8 r;
		u8 g;
		u8 b;
		u8 a;
	};
	
	struct Image {
		int width;
		int height;
		Pixel* data;
	};

// Globals.

	Image image_player_ship = {0};
	Image image_basic_enemy_ship = {0};
	Image image_strong_enemy_ship = {0};

// Functions.

	Pixel get_rgb (u8 r, u8 g, u8 b) {
		return (Pixel){ r, g, b, 255 };
	}
	Pixel get_rgba (u8 r, u8 g, u8 b, u8 a) {
		return (Pixel){ r, g, b, a };
	}
	// We also need these because Windows uses colors in BGRA format.
	Pixel get_bgr (u8 r, u8 g, u8 b) {
		return (Pixel){ b, g, r, 255 };
	}
	Pixel get_bgra (u8 r, u8 g, u8 b, u8 a) {
		return (Pixel){ b, g, r, a };
	}
	
	int create_image (Image* image, int width, int height) {
		Image newimage = {
			.width = width,
			.height = height,
			.data = malloc(width*height*sizeof(Pixel)),
		};
		*image = newimage;
		return 0;
	}
	int load_image_from_file (Image* image, char* path) {
		int width;
		int height;
		int channels;
		void* image_data = stbi_load(path, &width, &height, &channels, 4);
		if (!image_data) {
			printf("Failed to load image (%s)!\n", path);
			return 1;
		}
		
		Image newimage = {
			.width = width,
			.height = height,
			.data = image_data,
		};
		// Since Windows uses BGRA format, we need to swap the red and blue values on every pixel.
		int pixelcount = width * height;
		for (int i=0; i<pixelcount; i++) {
			u8 r = newimage.data[i].r;
			newimage.data[i].r = newimage.data[i].b;
			newimage.data[i].b = r;
		}
		
		*image = newimage;
		return 0;
	}
	void resize_image (Image* image, int width, int height) {
		// Note: this causes the pixels to move into wrong positions, but we'll just re-draw the image so it doesn't matter.
		image->width = width;
		image->height = height;
		image->data = realloc(image->data, width*height*sizeof(Pixel));
	}
	
	void fill_image_with_color (Image* target, Pixel color) {
		int pixelcount = target->width * target->height;
		for (int i=0; i<pixelcount; i++) {
			target->data[i] = color;
		}
	}
	void draw_rect (Image* target, int xstart, int ystart, int width, int height, Pixel color) {
		int xend = xstart + width;
		int yend = ystart + height;
		// Prevent overflow from right/bottom.
		if (xend > target->width) xend = target->width;
		if (yend > target->height) yend = target->height;
		// Prevent underflow from left/top.
		if (xstart < 0) xstart = 0;
		if (ystart < 0) ystart = 0;
		
		for (int y=ystart; y<yend; y++) {
			for (int x=xstart; x<xend; x++) {
				target->data[y*target->width+x] = color;
			}
		}
	}
	void draw_image (Image* target, Image* source, int target_x, int target_y) {
		// Crop the source image if necessary, we shouldn't draw outside of the target image.
		int crop_x = 0;
		int crop_y = 0;
		int crop_w = source->width;
		int crop_h = source->height;
		if (target_x < 0) {
			crop_w -= -target_x;
			crop_x += -target_x;
			target_x = 0;
		}
		if (target_y < 0) {
			crop_h -= -target_y;
			crop_y += -target_y;
			target_y = 0;
		}
		if (target_x+crop_w > target->width) {
			crop_w = target->width - target_x;
		}
		if (target_y+crop_h > target->height) {
			crop_h = target->height - target_y;
		}
		
		// Exit if the image ended up completely cropped out, this happens if you try to draw an image off-screen.
		if (crop_w <= 0 || crop_h <= 0) return;
		
		for (int y=0; y<crop_h; y++) {
			for (int x=0; x<crop_w; x++) {
				Pixel* dest = target->data + ((target_y+y)*target->width + target_x+x);
				Pixel* src = source->data + ((crop_y+y)*source->width + crop_x+x);
				
				// Only draw pixels that >0 alpha. You could do more advanced blending here to support semi-transparent pixels, but that's more complicated than we need.
				if (src->a) *dest = *src;
			}
		}
	}
