
// Types

	struct Window {
		int width;
		int height;
		HWND window_handle; // A "handle" that Windows uses to identify different windows. "Handles" in Windows are just pointers that point to some thing that only Windows understands.
		HDC	device_context; // Another handle that Windows uses for some graphics-related things.
	};

// Functions.

	// By calling this, Windows will send all the occurred events into our event handler function above.
	void handle_all_window_events () {
		MSG windowmessage;
		while (PeekMessageA(&windowmessage, NULL, 0, 0, PM_REMOVE)) {
			// This translates keyboard input events (WM_KEYDOWN) and creates text input events (WM_CHAR) which are extremely useful if you want to read text input. They're not used in this program though. I don't know why you need to call this function manually.
			TranslateMessage(&windowmessage);
			// Send the message, this will cause Windows to send information about the event to our callback window_event_handler().
			DispatchMessageA(&windowmessage);
		}
	}

	// Open a new window.
	int create_window (Window* window, int width, int height, char* titletext) {
		Window newwindow = {
			.width = width,
			.height = height,
		};
		
		// You must register a "window class" before you make a window. I don't know why, maybe it helps 2 different programs create similar windows or something... note that if you want to create multiple windows, you can't create 2 classes with the same lpszClassName. You must either create 1 class and use the same name in CreateWindowExA, or create a new class with different name.
		WNDCLASSA windowclass = {
			.style = CS_OWNDC | CS_HREDRAW | CS_VREDRAW,
			.lpfnWndProc = window_event_handler, // Our event handler function above.
			.hInstance = GetModuleHandle(NULL),
			.hCursor = LoadCursorA(NULL, IDC_ARROW),
			.lpszClassName = "some_window_class_name",
		};
		ATOM result = RegisterClassA(&windowclass);
		if (!result) {
			printf("Failed to register window class!\n");
			return 1;
		}
		
		// These can be used to set various properties, for example if you add WS_EX_TOPMOST to windowstylex, the window will always stay on top of other windows.
		DWORD windowstyle = WS_OVERLAPPEDWINDOW | WS_VISIBLE;
		DWORD windowstylex = 0;
		
		// Optional: fix window size. The size you provide to CreateWindow below includes the window border, so if you create a 200x200 window, the image on the inside will be smaller than that. Doing this will cause the image inside the window to be 200x200. Note that if you remove this, the window.width and window.height values will not represent the correct canvas size.
		RECT rect = {.right=width, .bottom=height};
		AdjustWindowRectEx(&rect, windowstyle, FALSE, windowstylex);
		width = rect.right - rect.left;
		height = rect.bottom - rect.top;
		
		// Create the window.
		newwindow.window_handle = CreateWindowExA(
			windowstylex,				// DWORD     dwExStyle
			windowclass.lpszClassName,	// LPCSTR    lpClassName
			titletext,					// LPCSTR    lpWindowName
			windowstyle,				// DWORD     dwStyle
			CW_USEDEFAULT,				// int       X
			CW_USEDEFAULT,				// int       Y
			width,						// int       nWidth
			height,						// int       nHeight
			NULL,						// HWND      hWndParent
			NULL,						// HMENU     hMenu
			GetModuleHandle(NULL),		// HINSTANCE hInstance
			NULL						// LPVOID    lpParam
		);
		if (!newwindow.window_handle) {
			printf("Failed to open window!\n");
			return 2;
		}
		
		// I'm not exactly sure why this is separate from the window handle. It's used for various graphics-related things.
		newwindow.device_context = GetDC(newwindow.window_handle);
		
		*window = newwindow;
		return 0;
	}

	// This makes Windows display an image on the window.
	void display_image_on_window (Window* window, Image* img) {
		BITMAPINFO bitmap_info = {
			.bmiHeader = {
				.biSize = sizeof(BITMAPINFOHEADER),
				.biWidth = img->width,
				.biHeight = -img->height, // Negative value = use top-down pixel order. If this is positive, the image will be upside-down.
				.biPlanes = 1,
				.biBitCount = 32,
				.biCompression = BI_RGB,
			}
		};
		StretchDIBits(
			window->device_context,
			0, 0, window->width, window->height,
			0, 0, img->width, img->height,
			img->data,
			&bitmap_info,
			DIB_RGB_COLORS,
			SRCCOPY
		);
	}
	
	// This isn't a very helpful function, but I always forget how to use SleepEx, so it's easier to wrap it inside my own function.
	void sleep_milliseconds (int time) {
		SleepEx(time, FALSE);
	}
