#ifndef LIB_BITMAP_FONT_H
#define LIB_BITMAP_FONT_H 1
#pragma once

/*
	!! NOTE !!  do not free the image that you use to generate this if you plan on using software rendering.
	
	// destination image
	Image dest = image_new(vec2i(200, 200));
	image_clear_with_color(&dest, PIXEL_RGBA8_GRAY30);
	
	// load fonts
	STRING_FROM_STRINGS(path_br, meta.program.path, string("resources/img/font_bigger.png"));
	STRING_FROM_STRINGS(path_bb, meta.program.path, string("resources/img/font_bigger_bold.png"));
	Bitmapfont font_br = bitmapfont_load_from_path(path_br, 5, 15, 5, 0);
	Bitmapfont font_bb = bitmapfont_load_from_path(path_bb, 5, 15, 5, -1);
	
	// draw text
	image_draw_bitmapfont_text(&dest, &font_br, vec2i(10, 10), string("Hello world 1 + 2 = 345\nJust like make game..!"));
	// draw boldable text
	image_draw_bitmapfont_boldable_text(&dest, &font_br, &font_bb, vec2i(10, 10), string("Hello world 1 + 2 = 345\nJust like \bmake game\b..!"));
	
	// save dest image as a bmp file
	ez_write_image_as_bmp_file(&dest, string("C:/Users/Sun/Desktop/test.bmp"));
*/

// things

	typedef struct {
		Bounds2i bounds;
		int yoffset;
	} Bitmapfontchar;
	
	typedef struct {
		Bitmapfontchar* characters;
		Image image;
		int maxcharwidth; // for drawing monospace text
		int baseline;
		int lineheight;
		int spacewidth;
		int letterspacing;
		f32 letterspacingf;
	} Bitmapfont;

// stuff

	static Bitmapfont bitmapfont_load_from_image (Image* image, int baseline, int lineheight, int spacewidth, f32 letterspacing) {
		if (!image || !image->data) {
			printf("Invalid image.\n");
			return (Bitmapfont){0};
		}
		
		Bitmapfont font = {0};
		font.characters = mem_alloc(sizeof(Bitmapfontchar)*256);
		font.baseline = baseline;
		font.lineheight = lineheight;
		font.spacewidth = spacewidth;
		font.letterspacing = round(letterspacing);
		font.letterspacingf = letterspacing;
		font.image = *image;
		
		const int COLUMNS = 16;
		int TILEWIDTH = image->size.w / COLUMNS;
		int TILEHEIGHT = image->size.h / COLUMNS;
		
		for (int row=0; row<COLUMNS; row++) {
			for (int col=0; col<COLUMNS; col++) {
				Bitmapfontchar c = {
					.bounds = {
						.x = col * TILEWIDTH,
						.y = row * TILEHEIGHT,
						.w = TILEWIDTH,
						.h = TILEHEIGHT,
					}
				};
				
				// loop through edge pixels to remove empty space from the sprite bounds
				Rgba8* pixel;
				
				int emptysize;
				// clip right
				emptysize = 0;
				for (int x=c.bounds.x+c.bounds.w-1; x>=c.bounds.x; x--) {
					for (int y=c.bounds.y; y<c.bounds.y+c.bounds.h; y++) {
						pixel = ((Rgba8*)image->data) + (y*image->size.w + x);
						if (pixel->a != 0) {
							goto CLIP_X_RIGHT;
						}
					}
					emptysize ++;
				}
				CLIP_X_RIGHT: {
					c.bounds.w -= emptysize;
				}
				// clip left
				emptysize = 0;
				for (int x=c.bounds.x; x<c.bounds.x+c.bounds.w; x++) {
					for (int y=c.bounds.y; y<c.bounds.y+c.bounds.h; y++) {
						pixel = ((Rgba8*)image->data) + (y*image->size.w + x);
						if (pixel->a != 0) {
							goto CLIP_X_LEFT;
						}
					}
					emptysize ++;
				}
				CLIP_X_LEFT: {
					c.bounds.x += emptysize;
					c.bounds.w -= emptysize;
				}
				// clip bottom
				emptysize = 0;
				for (int y=c.bounds.y+c.bounds.h-1; y>=c.bounds.y; y--) {
					for (int x=c.bounds.x; x<c.bounds.x+c.bounds.w; x++) {
						pixel = ((Rgba8*)image->data) + (y*image->size.w + x);
						if (pixel->a != 0) {
							goto CLIP_Y_BOTTOM;
						}
					}
					emptysize ++;
				}
				CLIP_Y_BOTTOM: {
					c.bounds.h -= emptysize;
				}
				// clip top
				emptysize = 0;
				for (int y=c.bounds.y; y<c.bounds.y+c.bounds.h; y++) {
					for (int x=c.bounds.x; x<c.bounds.x+c.bounds.w; x++) {
						pixel = ((Rgba8*)image->data) + (y*image->size.w + x);
						if (pixel->a != 0) {
							goto CLIP_Y_TOP;
						}
					}
					emptysize ++;
				}
				CLIP_Y_TOP: {
					c.bounds.y += emptysize;
					c.bounds.h -= emptysize;
					c.yoffset = emptysize;
				}
				
				if (row*COLUMNS+col == ' ') {
					if (c.bounds.w <= 0) c.bounds.w = spacewidth;
				}
				
				font.characters[row*COLUMNS+col] = c;
				font.maxcharwidth = max(font.maxcharwidth, c.bounds.w);
			}
		}
		
		return font;
	}
	static Bitmapfont bitmapfont_load_from_path (char* path, int baseline, int lineheight, int spacewidth, f32 letterspacing) {
		Image image;
		int e = image_load_from_path(&image, path);
		if (e) {
			printf("Failed to load image.\n");
			return (Bitmapfont){0};
		}
		
		Bitmapfont font = bitmapfont_load_from_image(&image, baseline, lineheight, spacewidth, letterspacing);
		return font;
	}
	
	static void bitmapfont_free (Bitmapfont* font) {
		if (font->characters) mem_free(font->characters);
		image_free(&font->image);
		*font = (Bitmapfont){0};
	}
	
	static Vec2i image_draw_bitmapfont (Image* target, Bitmapfont* regular, Bitmapfont* bold, Vec2i pos, char* text) {
		if (!regular) {
			return (Vec2i){0};
		}
		Vec2i size = pos;
		Bitmapfont* font = regular;
		char* c = text;
		int text_length = strlen(text);
		int x = pos.x;
		int y = pos.y;
		for (int i=0; i<text_length; i++, c++) {
			// toggle bold
			if (*c == '\b' && bold) {
				font = (font == regular) ? bold : regular;
				continue;
			}
			// handle new line in a special way
			if (*c == '\n') {
				if (x > size.x) size.x = x;
				x = pos.x;
				y += font->lineheight;
				continue;
			}
			Bounds2i b = font->characters[(u8)*c].bounds;
			int yoff = font->characters[(u8)*c].yoffset;
			if (target) image_draw_image_cropped(target, &font->image, b, vec2i(x, y+yoff), 1);
			x += b.w + font->letterspacing;
		}
		if (x > size.x) size.x = x;
		size.x -= pos.x;
		size.y = y + font->lineheight - pos.y;
		return size;
	}
	static Vec2f image_draw_bitmapfont_smoothly (Image* target, Bitmapfont* regular, Bitmapfont* bold, Vec2f pos, Vec2f zoom, char* text) {
		if (!regular) {
			return (Vec2f){0};
		}
		Vec2f size = pos;
		Bitmapfont* font = regular;
		char* c = text;
		int text_length = strlen(text);
		f32 x = pos.x;
		f32 y = pos.y;
		for (int i=0; i<text_length; i++, c++) {
			// toggle bold
			if (*c == '\b' && bold) {
				font = (font == regular) ? bold : regular;
				continue;
			}
			// handle new line in a special way
			if (*c == '\n') {
				if (x > size.x) size.x = x;
				x = pos.x;
				y += (f32)font->lineheight * zoom.y;
				continue;
			}
			Bounds2i b = font->characters[(u8)*c].bounds;
			f32 yoff = (f32)font->characters[(u8)*c].yoffset * zoom.y;
			if (target) image_draw_image_smoothly_cropped(target, &font->image, b, vec2f(x, y+yoff), zoom);
			x += (b.w + (font->letterspacingf)) * zoom.x;
		}
		if (x > size.x) size.x = x;
		size.x -= pos.x;
		size.y = y + ((f32)font->lineheight * zoom.y) - pos.y;
		return size;
	}

// include guard
#endif