#pragma once
/*
	Virtual MEMory allocator. (temp/area allocator)
	- This is slightly more "advanced" (simpler but uses newer concepts) than usual allocators and arrays, because rather than just allocating and re-allocating memory, this pre-reserves virtual addess space and never actually has to re-allocate memory when it grows.
	
	----------
	
	vmem_ez_init(1024*1024*1024, 1024*1024); // 1 GB of maximum potential memory, start out with 1 MB.
	
	void* data = vmem_alloc(sizeof(Somestruct)*50);
	String str = vmem_string_from_print("There's %i bananas in there.", 150);
	String str = vmem_string_from_strings_local(string("/icons/image.png"));
	
	vmem_set_pos(0); // Do this to reset all the memory.
	
	----------
	
	String foo = vmem_string_from_strings_local(string("/res/"));
	
	u64 pos = vmem_get_pos(); // Store the buffer position.
	for (int i=0; i<100; i++) {
		String bar = vmem_string_from_strings(foo, string("bar.png"));
	}
	
	vmem_set_pos(pos); // Restore position, this will effectively "free" everything that was allocated after the position was retrieved.
	for (int i=0; i<100; i++) {
		String bar = vmem_string_from_strings(foo, string("bar.png"));
	}
	
	for (int i=0; i<10000000; i++) {
		String bar = vmem_string_from_print("%s/bar_%i.png", foo.data, i);
		vmem_set_pos(pos); // Doing this in the loop is much more effective. Builds up less memory, and is much more cache local.
	}
	
	vmem_set_pos(0); // "Free" all the memory.
*/

// types

	typedef struct {
		void* data;
		u64 used; // Amount of committed memory that has been consumed by allocations.
		u64 committed; // Amount of memory committed and usable.
		u64 reserved; // Amount of virtual address space reserved.
		
		void* latest; // Latest allocation, used for fast realloc.
	} Vmem;
	
	Vmem _vmem_default = {0};
	Vmem* _vmem_active = NULL;

// functionality

	static Vmem vmem_new (u64 absolute_max_cap, u64 min_memory) START
		ASSERT(min_memory > 0);
		ASSERT(absolute_max_cap > min_memory*2ll);
		
		u64 page_size = os_mem_get_page_size();
		ASSERT(page_size);
		min_memory = (min_memory+page_size-1ll) / page_size * page_size;
		absolute_max_cap = (absolute_max_cap+page_size-1ll) / page_size * page_size;
		ASSERT(min_memory);
		ASSERT(absolute_max_cap);
		
		Vmem vmem = {0};
		vmem.reserved = absolute_max_cap;
		vmem.committed = min_memory;
		vmem.data = os_mem_reserve_virtual_address_space(absolute_max_cap);
		if (!vmem.data) {
			// Sigh
		}
		Errnum e = os_mem_commit_virtual_memory(vmem.data, min_memory);
		if (e) {
			// Sigh
		}
		
		RETURN(vmem);
	END
	static void vmem_free (Vmem* vmem) START
		ASSERT(vmem);
		ASSERT(vmem->data);
		
		os_mem_free_virtual_address_space(vmem->data);
		
		*vmem = (Vmem){0};
	END
	static void vmem_ez_init (u64 absolute_max_cap, u64 min_memory) {
		_vmem_default = vmem_new(absolute_max_cap, min_memory);
		_vmem_active = &_vmem_default;
	}
	
	static u64 _vmem_get_pos (Vmem* vmem) START
		ASSERT(vmem);
		
		RETURN(vmem->used);
	END
	static void _vmem_set_pos (Vmem* vmem, u64 pos) START
		ASSERT(vmem);
		ASSERT(pos >= 0);
		
		vmem->used = pos;
		vmem->latest = NULL;
	END
	static void _vmem_expand_committed (Vmem* vmem, u64 desired) START
		ASSERT(desired >= 0);
		
		desired += vmem->used;
		if (vmem->used + desired > vmem->reserved) {
			lprintf_error("Vmem ran out of space (not enough virtual address space has been reserved).\n");
			RETURN();
		}
		
		#if !BUILD_RELEASE
			lprintf_dev("Vmem committed memory ran out (%i MB), expanding...\n", vmem->committed/1024ll/1024ll);
		#endif
		
		u64 goal = vmem->committed;
		while (goal < desired) {
			goal *= 2ll;
		}
		
		Errnum e = os_mem_commit_virtual_memory(vmem->data, goal);
		if (e) {
			lprintf_error("Vmem failed to commit more memory (system memory ran out?).\n");
			// What happens when allocation fails? Does the whole thing get screwed, or does already-reserved/allocated stuff remain?
			// os_mem_free_virtual_address_space(vmem->data);
			// *vmem = (Vmem){0};
			RETURN();
		}
		
		vmem->committed = goal;
	END
	
	static u64 _vmem_get_alloc_size (Vmem* vmem, void* mem) START
		ASSERT(vmem);
		ASSERT(vmem->data);
		ASSERT(mem >= vmem->data+sizeof(u64));
		
		RETURN(((u64*)mem)[-1]);
	END
	static void* _vmem_alloc (Vmem* vmem, u64 amount) START
		ASSERT(vmem);
		ASSERT(vmem->data);
		ASSERT(amount > 0);
		
		amount += ((amount+(8-1))/8)*8; // Align to nearest 8 bytes. This keeps all allocations and the size value aligned, and also helps with reallocations a bit.
		amount += sizeof(u64);
		
		if (vmem->used + amount > vmem->committed) {
			_vmem_expand_committed(vmem, amount);
			if (vmem->used + amount > vmem->committed) {
				RETURN(NULL);
			}
		}
		
		void* cursor = vmem->data + vmem->used;
		*(u64*)cursor = amount-sizeof(u64); // Store the size of vmem allocated block, so it can be "reallocated" (need to know how much data to move).
		cursor += sizeof(u64);
		
		vmem->used += amount;
		vmem->latest = cursor;
		
		RETURN(cursor);
	END
	static void* _vmem_realloc (Vmem* vmem, void* mem, u64 amount) START
		ASSERT(mem);
		ASSERT(amount > 0);
		
		u64* oldamount = mem-sizeof(u64);
		if (*oldamount >= amount) {
			RETURN(mem);
		}
		
		if (vmem->latest == mem) { // Reallocating the latest allocation, just expand it instead of moving data.
			ASSERT(vmem->used >= *oldamount);
			vmem->used -= *oldamount;
			if (vmem->used + amount > vmem->committed) {
				_vmem_expand_committed(vmem, amount);
				if (vmem->used + amount > vmem->committed) {
					RETURN(NULL);
				}
			}
			vmem->used += amount;
			*oldamount = amount;
			RETURN(mem);
		}
		
		char* cursor = _vmem_alloc(vmem, amount);
		if (cursor) mem_copy(cursor, *oldamount, mem);
		RETURN(cursor);
	END
	static void* _vmem_alloc_zeroed (Vmem* vmem, u64 amount) {
		ASSERT(amount > 0);
		
		void* cursor = _vmem_alloc(vmem, amount);
		if (cursor) mem_zero(amount, cursor);
		return cursor;
	}
	static void* _vmem_alloc_data (Vmem* vmem, void* data, u64 amount) {
		ASSERT(amount > 0);
		
		char* cursor = _vmem_alloc(vmem, amount);
		if (cursor) mem_copy(cursor, amount, data);
		return cursor;
	}
	static void* _vmem_alloc_print (Vmem* vmem, char* str, ...) START
		ASSERT(vmem);
		ASSERT(vmem->data);
		ASSERT(str);
		
		void* data = vmem->data + vmem->used;
		u64 amount = STB_SPRINTF_MIN + sizeof(u64);
		if (vmem->used + amount > vmem->committed) {
			_vmem_expand_committed(vmem, amount);
			if (vmem->used + amount > vmem->committed) {
				RETURN(NULL);
			}
		}
		amount = sizeof(u64);
		
		char* callback (char* buf, void* user, int len) {
			amount += len;
			if (vmem->used + amount > vmem->committed) {
				_vmem_expand_committed(vmem, amount);
				if (vmem->used + amount > vmem->committed) {
					amount = 0;
					RETURN(NULL);
				}
			}
			return data + amount;
		}
		
		va_list va;
		va_start(va, str);
		stbsp_vsprintfcb((char*(*)(const char* buf, void* user, int len))callback, NULL, data+amount, (char const*)str, va);
		va_end(va);
		
		if (!amount) RETURN(NULL);
		
		amount -= sizeof(u64);
		u64 string_length = amount;
		amount += 1; // Null.
		amount += ((amount+(8-1))/8)*8; // Align to nearest 8 bytes. This keeps all allocations and the size value aligned, and also helps with reallocations a bit.
		*(u64*)(data) = amount;
		data += sizeof(u64);
		vmem->used += sizeof(u64);
		vmem->used += amount;
		mem_zero(amount-string_length, data+string_length); // Zero the padding. TODO: make sure space for this is allocated.
		RETURN(data);
	END
	
	static String _vmem_alloc_string (Vmem* vmem, String str) {
		ASSERT(str.length);
		ASSERT(str.data);
		
		void* cursor = _vmem_alloc(vmem, str.length+1);
		if (cursor) {
			mem_copy(cursor, str.length, str.data);
			*(u8*)(cursor+str.length) = 0;
			str.data = cursor;
		}
		else {
			str = (String){0};
		}
		return str;
	}
	static String _vmem_alloc_string_from_cstr (Vmem* vmem, char* str) {
		ASSERT(str);
		
		u64 len = strlen(str);
		void* cursor = _vmem_alloc(vmem, len+1);
		if (cursor) {
			mem_copy(cursor, len, str);
			*(u8*)(cursor+len) = 0;
			return (String){.data=cursor, .length=len};
		}
		return ZERO(String);
	}
	static String _vmem_string_from_print (Vmem* vmem, char* str, ...) START
		ASSERT(vmem);
		ASSERT(vmem->data);
		ASSERT(str);
		
		void* data = vmem->data + vmem->used;
		u64 amount = STB_SPRINTF_MIN + sizeof(u64);
		if (vmem->used + amount > vmem->committed) {
			_vmem_expand_committed(vmem, amount);
			if (vmem->used + amount > vmem->committed) {
				RETURN(ZERO(String));
			}
		}
		amount = sizeof(u64);
		
		char* callback (char* buf, void* user, int len) {
			amount += len;
			if (vmem->used + amount > vmem->committed) {
				_vmem_expand_committed(vmem, amount);
				if (vmem->used + amount > vmem->committed) {
					amount = 0;
					RETURN(NULL);
				}
			}
			return data + amount;
		}
		
		va_list va;
		va_start(va, str);
		stbsp_vsprintfcb((char*(*)(const char* buf, void* user, int len))callback, NULL, data+amount, (const char*)str, va);
		va_end(va);
		
		if (!amount) RETURN(ZERO(String));
		
		amount -= sizeof(u64);
		u64 string_length = amount;
		amount += 1; // Null.
		amount += ((amount+(8-1))/8)*8; // Align to nearest 8 bytes. This keeps all allocations and the size value aligned, and also helps with reallocations a bit.
		*(u64*)(data) = amount;
		data += sizeof(u64);
		vmem->used += sizeof(u64);
		vmem->used += amount;
		mem_zero(amount-string_length, data+string_length); // Zero the padding. TODO: make sure space for this is allocated.
		RETURN((String){.length=string_length, .data=data});
	END
	static String _vmem_string_from_strings (Vmem* vmem, String* list) START
		ASSERT(list);
		ASSERT(list[0].data);
		
		u64 length = 0;
		u64 i = 0;
		while (list[i].data) {
			length += list[i].length;
			i ++;
		}
		
		String str = {0};
		void* cursor = _vmem_alloc(vmem, length+1);
		if (cursor) {
			str.data = cursor;
			str.length = length;
			
			i = 0;
			while (list[i].data) {
				ASSERT(list[i].data);
				mem_copy(cursor, list[i].length, list[i].data);
				cursor += list[i].length;
				i ++;
			}
			str.data[length] = 0;
		}
		RETURN(str);
	END
	
	#define vmem_reset()								_vmem_reset(_vmem_active)
	#define vmem_get_pos()								_vmem_get_pos(_vmem_active)
	#define vmem_set_pos(pos)							_vmem_set_pos(_vmem_active, pos)
	
	#define vmem_get_alloc_size(mem)					_vmem_get_alloc_size(_vmem_active, mem)
	#define vmem_alloc(amount)							_vmem_alloc(_vmem_active, amount)
	#define vmem_realloc(mem, amount)					_vmem_realloc(_vmem_active, mem, amount)
	#define vmem_alloc_zeroed(amount)					_vmem_alloc_zeroed(_vmem_active, amount)
	#define vmem_alloc_data(data, amount)				_vmem_alloc_data(_vmem_active, data, amount)
	#define vmem_alloc_print(str, ...)					_vmem_alloc_print(_vmem_active, str, __VA_ARGS__)
	
	#define vmem_alloc_string(str)						_Generic(str, String:_vmem_alloc_string, char*:_vmem_alloc_string_from_cstr)(_vmem_active, str)
	#define vmem_string_from_print(...)					_vmem_string_from_print(_vmem_active, __VA_ARGS__)
	#define vmem_string_from_strings(...)				_vmem_string_from_strings(_vmem_active, ((String[]){__VA_ARGS__,(String){0}}))
	#define vmem_string_from_strings_local(...)			_vmem_string_from_strings(_vmem_active, ((String[]){os_get_program_path(),__VA_ARGS__,(String){0}}))
